// Excel处理Worker
// 使用本地XLSX库
importScripts('./xlsx.full.min.js');

self.onmessage = function(e) {
  try {
    const { data, type, neededColumns } = e.data;
    
    if (type === 'parseExcel') {
      const result = parseExcelData(data, neededColumns);
      self.postMessage({
        type: 'success',
        data: result
      });
    }
  } catch (error) {
    console.error('Worker error:', error);
    self.postMessage({
      type: 'error',
      error: error.message || '未知错误'
    });
  }
};

function parseExcelData(arrayBuffer, neededColumns = null) {
  try {
    if (!arrayBuffer) {
      throw new Error('没有接收到数据');
    }
    
    const data = new Uint8Array(arrayBuffer);
    const workbook = XLSX.read(data, { 
      type: 'array',
      cellDates: true,
      cellNF: false,
      cellStyles: false,
      cellHTML: false,
      cellFormula: false,
      cellText: false
    });
    
    if (!workbook.SheetNames || workbook.SheetNames.length === 0) {
      throw new Error('Excel文件中没有找到工作表');
    }
    
    const sheetName = workbook.SheetNames[0];
    const worksheet = workbook.Sheets[sheetName];
    
    if (!worksheet) {
      throw new Error('无法读取工作表数据');
    }
    
    // 限制最大行数，防止内存溢出
    const maxRows = 100000; // 增加到10万行
    const range = XLSX.utils.decode_range(worksheet['!ref'] || 'A1');
    if (range.e.r > maxRows) {
      range.e.r = maxRows - 1;
      worksheet['!ref'] = XLSX.utils.encode_range(range);
    }
    
    const jsonData = XLSX.utils.sheet_to_json(worksheet, {
      header: 1,
      defval: '',
      blankrows: false
    });
    
    if (!jsonData || jsonData.length === 0) {
      throw new Error('Excel文件中没有数据');
    }
    
    // 转换为对象数组
    const headers = jsonData[0];
    if (!headers || headers.length === 0) {
      throw new Error('Excel文件中没有列标题');
    }
    
    const rows = jsonData.slice(1).map(row => {
      const obj = {};
      headers.forEach((header, index) => {
        obj[header] = row[index] || '';
      });
      return obj;
    });
    
    // 如果指定了需要的列，则只保留这些列
    if (neededColumns && neededColumns.length > 0) {
      return rows.map(row => {
        const optimizedRow = {};
        neededColumns.forEach(col => {
          optimizedRow[col] = row[col] || '';
        });
        return optimizedRow;
      });
    }
    
    return rows;
  } catch (error) {
    console.error('Excel解析错误:', error);
    throw new Error('Excel解析失败: ' + error.message);
  }
}
