# Bug修复说明

## 问题描述
**Bug**: 点击单个文件"开始"下载时，会下载所有文件

## 问题原因
在 `startDownload` 方法的 `finally` 块中，当单个文件下载完成时，会调用 `startNextPendingDownload()` 方法，该方法会自动启动其他等待中的下载任务。

```javascript
// 问题代码
finally {
  activeDownloadCount.value--
  // 这里会启动其他等待中的任务
  if (!isBatchDownloading.value) {
    startNextPendingDownload()
  }
}
```

## 修复方案

### 1. 添加单个下载标志
在下载store中添加 `isSingleDownloading` 标志来区分单个下载和批量下载：

```javascript
const isSingleDownloading = ref(false) // 是否正在单个下载
```

### 2. 修改自动启动逻辑
修改 `startNextPendingDownload` 方法，在单个下载时也不自动启动其他任务：

```javascript
const startNextPendingDownload = () => {
  // 如果正在批量下载或单个下载，不自动启动其他任务
  if (isBatchDownloading.value || isSingleDownloading.value) return
  
  const pendingDownload = downloads.value.find(d => d.status === 'pending')
  if (pendingDownload && activeDownloadCount.value < maxConcurrentDownloads.value) {
    startDownload(pendingDownload.id)
  }
}
```

### 3. 修改单个下载调用
创建专门的单个下载方法，设置单个下载标志：

```javascript
// 单个文件下载
const startSingleDownload = async (downloadId) => {
  downloadStore.isSingleDownloading = true
  try {
    await downloadStore.startDownload(downloadId)
  } finally {
    downloadStore.isSingleDownloading = false
  }
}

// 单个文件恢复下载
const resumeSingleDownload = async (downloadId) => {
  downloadStore.isSingleDownloading = true
  try {
    await downloadStore.resumeDownload(downloadId)
  } finally {
    downloadStore.isSingleDownloading = false
  }
}
```

### 4. 更新按钮事件
将表格中的单个下载按钮事件改为调用新的方法：

```javascript
// 开始按钮
@click="startSingleDownload(row.id)"

// 继续按钮
@click="resumeSingleDownload(row.id)"

// 重试按钮
@click="startSingleDownload(row.id)"
```

## 修复后的行为

### ✅ 正确的行为
- **单个下载**：点击单个文件的"开始"按钮，只下载该文件
- **批量下载**：点击"批量开始选中"或"下载当前页"，只下载选中的文件
- **自动启动**：只有在非批量下载且非单个下载时，才会自动启动其他等待中的任务

### ❌ 错误的行为（已修复）
- 点击单个文件"开始"按钮，所有等待中的文件都开始下载

## 测试方法

### 测试单个下载
1. 上传Excel文件，解析添加多个下载任务
2. 只点击其中一个任务的"开始"按钮
3. **验证结果**：只有该任务开始下载，其他任务保持"等待"状态

### 测试批量下载
1. 选择多个任务
2. 点击"批量开始选中"按钮
3. **验证结果**：只有选中的任务开始下载

### 测试自动启动
1. 确保没有进行批量或单个下载操作
2. 手动开始一个下载任务
3. **验证结果**：其他等待中的任务可能会自动开始（这是正常的多线程下载行为）

## 技术细节

### 下载模式区分
- **单个下载模式**：`isSingleDownloading = true`
- **批量下载模式**：`isBatchDownloading = true`
- **自动启动模式**：两个标志都为 `false`

### 标志管理
- 单个下载：在下载开始时设置 `isSingleDownloading = true`，下载完成后重置
- 批量下载：在批量操作开始时设置 `isBatchDownloading = true`，批量操作完成后重置
- 自动启动：只有在两个标志都为 `false` 时才允许自动启动其他任务

## 影响范围

### 修复的功能
- ✅ 单个文件下载
- ✅ 单个文件恢复下载
- ✅ 单个文件重试下载
- ✅ 批量下载（不受影响）
- ✅ 自动启动（在适当时候仍然工作）

### 不受影响的功能
- 暂停下载
- 删除下载
- 状态筛选
- 分页显示
- 断点续传

## 验证清单

- [ ] 单个文件"开始"按钮只下载该文件
- [ ] 单个文件"继续"按钮只恢复该文件
- [ ] 单个文件"重试"按钮只重试该文件
- [ ] 批量下载功能正常工作
- [ ] 自动启动功能在适当时候仍然工作
- [ ] 其他功能不受影响
