'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SEVERITY = undefined;
exports.getDiagnostics = getDiagnostics;
exports.validateQuery = validateQuery;
exports.getRange = getRange;

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

var _graphql = require('graphql');

var _graphqlLanguageServiceParser = require('@apollographql/graphql-language-service-parser');

var _graphqlLanguageServiceUtils = require('@apollographql/graphql-language-service-utils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 *  Copyright (c) Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the license found in the
 *  LICENSE file in the root directory of this source tree.
 *
 *  
 */

var SEVERITY = exports.SEVERITY = {
  ERROR: 1,
  WARNING: 2,
  INFORMATION: 3,
  HINT: 4
};

function getDiagnostics(query) {
  var schema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
  var customRules = arguments[2];
  var isRelayCompatMode = arguments[3];

  var ast = null;
  try {
    ast = (0, _graphql.parse)(query);
  } catch (error) {
    var range = getRange(error.locations[0], query);
    return [{
      severity: SEVERITY.ERROR,
      message: error.message,
      source: 'GraphQL: Syntax',
      range: range
    }];
  }

  return validateQuery(ast, schema, customRules, isRelayCompatMode);
}

function validateQuery(ast) {
  var schema = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
  var customRules = arguments[2];
  var isRelayCompatMode = arguments[3];

  // We cannot validate the query unless a schema is provided.
  if (!schema) {
    return [];
  }

  var validationErrorAnnotations = mapCat((0, _graphqlLanguageServiceUtils.validateWithCustomRules)(schema, ast, customRules, isRelayCompatMode), function (error) {
    return annotations(error, SEVERITY.ERROR, 'Validation');
  });
  // Note: findDeprecatedUsages was added in graphql@0.9.0, but we want to
  // support older versions of graphql-js.
  var deprecationWarningAnnotations = !_graphql.findDeprecatedUsages ? [] : mapCat((0, _graphql.findDeprecatedUsages)(schema, ast), function (error) {
    return annotations(error, SEVERITY.WARNING, 'Deprecation');
  });
  return validationErrorAnnotations.concat(deprecationWarningAnnotations);
}

// General utility for map-cating (aka flat-mapping).
function mapCat(array, mapper) {
  return Array.prototype.concat.apply([], array.map(mapper));
}

function annotations(error, severity, type) {
  if (!error.nodes) {
    return [];
  }
  return error.nodes.map(function (node) {
    var highlightNode = node.kind !== 'Variable' && node.name ? node.name : node.variable ? node.variable : node;

    (0, _assert2.default)(error.locations, 'GraphQL validation error requires locations.');
    var loc = error.locations[0];
    var highlightLoc = getLocation(highlightNode);
    var end = loc.column + (highlightLoc.end - highlightLoc.start);
    return {
      source: 'GraphQL: ' + type,
      message: error.message,
      severity: severity,
      range: new _graphqlLanguageServiceUtils.Range(new _graphqlLanguageServiceUtils.Position(loc.line - 1, loc.column - 1), new _graphqlLanguageServiceUtils.Position(loc.line - 1, end))
    };
  });
}

function getRange(location, queryText) {
  var parser = (0, _graphqlLanguageServiceParser.onlineParser)();
  var state = parser.startState();
  var lines = queryText.split('\n');

  (0, _assert2.default)(lines.length >= location.line, 'Query text must have more lines than where the error happened');

  var stream = null;

  for (var i = 0; i < location.line; i++) {
    stream = new _graphqlLanguageServiceParser.CharacterStream(lines[i]);
    while (!stream.eol()) {
      var style = parser.token(stream, state);
      if (style === 'invalidchar') {
        break;
      }
    }
  }

  (0, _assert2.default)(stream, 'Expected Parser stream to be available.');

  var line = location.line - 1;
  var start = stream.getStartOfToken();
  var end = stream.getCurrentPosition();

  return new _graphqlLanguageServiceUtils.Range(new _graphqlLanguageServiceUtils.Position(line, start), new _graphqlLanguageServiceUtils.Position(line, end));
}

/**
 * Get location info from a node in a type-safe way.
 *
 * The only way a node could not have a location is if we initialized the parser
 * (and therefore the lexer) with the `noLocation` option, but we always
 * call `parse` without options above.
 */
function getLocation(node) {
  var typeCastedNode = node;
  var location = typeCastedNode.loc;
  (0, _assert2.default)(location, 'Expected ASTNode to have a location.');
  return location;
}