"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const graphql_1 = require("graphql");
const utils_1 = require("../../utils");
exports.providesNotOnEntity = ({ schema }) => {
    const errors = [];
    const types = schema.getTypeMap();
    for (const [typeName, namedType] of Object.entries(types)) {
        if (!graphql_1.isObjectType(namedType))
            continue;
        for (const [fieldName, field] of Object.entries(namedType.getFields())) {
            const serviceName = field.federation && field.federation.serviceName;
            if (!serviceName &&
                field.federation &&
                field.federation.provides &&
                !field.federation.belongsToValueType)
                throw Error('Internal Consistency Error: field with provides information does not have service name.');
            if (!serviceName)
                continue;
            const getBaseType = (type) => graphql_1.isListType(type) || graphql_1.isNonNullType(type)
                ? getBaseType(type.ofType)
                : type;
            const baseType = getBaseType(field.type);
            if (field.federation && field.federation.provides) {
                if (!graphql_1.isObjectType(baseType)) {
                    errors.push(utils_1.errorWithCode('PROVIDES_NOT_ON_ENTITY', utils_1.logServiceAndType(serviceName, typeName, fieldName) +
                        `uses the @provides directive but \`${typeName}.${fieldName}\` returns \`${field.type}\`, which is not an Object or List type. @provides can only be used on Object types with at least one @key, or Lists of such Objects.`));
                    continue;
                }
                const fieldType = types[baseType.name];
                const selectedFieldIsEntity = fieldType.federation && fieldType.federation.keys;
                if (!selectedFieldIsEntity) {
                    errors.push(utils_1.errorWithCode('PROVIDES_NOT_ON_ENTITY', utils_1.logServiceAndType(serviceName, typeName, fieldName) +
                        `uses the @provides directive but \`${typeName}.${fieldName}\` does not return a type that has a @key. Try adding a @key to the \`${baseType}\` type.`));
                }
            }
        }
    }
    return errors;
};
//# sourceMappingURL=providesNotOnEntity.js.map