'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getHoverInformation = getHoverInformation;

var _getAutocompleteSuggestions = require('./getAutocompleteSuggestions');

var _graphql = require('graphql');

/**
 *  Copyright (c) Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the license found in the
 *  LICENSE file in the root directory of this source tree.
 *
 *  
 */

/**
 * Ported from codemirror-graphql
 * https://github.com/graphql/codemirror-graphql/blob/master/src/info.js
 */

function getHoverInformation(schema, queryText, cursor, contextToken) {
  var token = contextToken || (0, _getAutocompleteSuggestions.getTokenAtPosition)(queryText, cursor);

  if (!schema || !token || !token.state) {
    return [];
  }

  var state = token.state;
  var kind = state.kind;
  var step = state.step;
  var typeInfo = (0, _getAutocompleteSuggestions.getTypeInfo)(schema, token.state);
  var options = { schema: schema };

  // Given a Schema and a Token, produce the contents of an info tooltip.
  // To do this, create a div element that we will render "into" and then pass
  // it to various rendering functions.
  if (kind === 'Field' && step === 0 && typeInfo.fieldDef || kind === 'AliasedField' && step === 2 && typeInfo.fieldDef) {
    var into = [];
    renderField(into, typeInfo, options);
    renderDescription(into, options, typeInfo.fieldDef);
    return into.join('').trim();
  } else if (kind === 'Directive' && step === 1 && typeInfo.directiveDef) {
    var _into = [];
    renderDirective(_into, typeInfo, options);
    renderDescription(_into, options, typeInfo.directiveDef);
    return _into.join('').trim();
  } else if (kind === 'Argument' && step === 0 && typeInfo.argDef) {
    var _into2 = [];
    renderArg(_into2, typeInfo, options);
    renderDescription(_into2, options, typeInfo.argDef);
    return _into2.join('').trim();
  } else if (kind === 'EnumValue' && typeInfo.enumValue && typeInfo.enumValue.description) {
    var _into3 = [];
    renderEnumValue(_into3, typeInfo, options);
    renderDescription(_into3, options, typeInfo.enumValue);
    return _into3.join('').trim();
  } else if (kind === 'NamedType' && typeInfo.type && typeInfo.type.description) {
    var _into4 = [];
    renderType(_into4, typeInfo, options, typeInfo.type);
    renderDescription(_into4, options, typeInfo.type);
    return _into4.join('').trim();
  }
}

function renderField(into, typeInfo, options) {
  renderQualifiedField(into, typeInfo, options);
  renderTypeAnnotation(into, typeInfo, options, typeInfo.type);
}

function renderQualifiedField(into, typeInfo, options) {
  if (!typeInfo.fieldDef) {
    return;
  }
  var fieldName = typeInfo.fieldDef.name;
  if (fieldName.slice(0, 2) !== '__') {
    renderType(into, typeInfo, options, typeInfo.parentType);
    text(into, '.');
  }
  text(into, fieldName);
}

function renderDirective(into, typeInfo, options) {
  if (!typeInfo.directiveDef) {
    return;
  }
  var name = '@' + typeInfo.directiveDef.name;
  text(into, name);
}

function renderArg(into, typeInfo, options) {
  if (typeInfo.directiveDef) {
    renderDirective(into, typeInfo, options);
  } else if (typeInfo.fieldDef) {
    renderQualifiedField(into, typeInfo, options);
  }

  if (!typeInfo.argDef) {
    return;
  }

  var name = typeInfo.argDef.name;
  text(into, '(');
  text(into, name);
  renderTypeAnnotation(into, typeInfo, options, typeInfo.inputType);
  text(into, ')');
}

function renderTypeAnnotation(into, typeInfo, options, t) {
  text(into, ': ');
  renderType(into, typeInfo, options, t);
}

function renderEnumValue(into, typeInfo, options) {
  if (!typeInfo.enumValue) {
    return;
  }
  var name = typeInfo.enumValue.name;
  renderType(into, typeInfo, options, typeInfo.inputType);
  text(into, '.');
  text(into, name);
}

function renderType(into, typeInfo, options, t) {
  if (!t) {
    return;
  }
  if (t instanceof _graphql.GraphQLNonNull) {
    renderType(into, typeInfo, options, t.ofType);
    text(into, '!');
  } else if (t instanceof _graphql.GraphQLList) {
    text(into, '[');
    renderType(into, typeInfo, options, t.ofType);
    text(into, ']');
  } else {
    text(into, t.name);
  }
}

function renderDescription(into, options, def) {
  if (!def) {
    return;
  }
  var description = typeof def.description === 'string' ? def.description : null;
  if (description) {
    text(into, '\n\n');
    text(into, description);
  }
  renderDeprecation(into, options, def);
}

function renderDeprecation(into, options, def) {
  if (!def) {
    return;
  }
  var reason = typeof def.deprecationReason === 'string' ? def.deprecationReason : null;
  if (!reason) {
    return;
  }
  text(into, '\n\n');
  text(into, 'Deprecated: ');
  text(into, reason);
}

function text(into, content) {
  into.push(content);
}