const system = require("../../../system");
const Dao = require("../../dao.base");

class StOrderItemDao extends Dao {
  constructor() {
    super(Dao.getModelName(StOrderItemDao));
  }

  async byIds(params) {
    let attrs = params.attrs || "*";
    let ids = params.ids;
    if (!ids || ids.length == 0) {
      return [];
    }

    let sql = [];
    sql.push("SELECT");
    sql.push(attrs);
    sql.push("FROM");
    sql.push(this.model.tableName);
    sql.push("WHERE id IN (:ids) ");
    if (params.trade_statuses && params.trade_statuses.length > 0) {
      sql.push("AND trade_status IN (:trade_statuses)");
    }
    if (params.saas_merchant_id) {
      sql.push("AND saas_merchant_id IN (:saas_merchant_id)");
    }
    if (params.unInvoice) {
      sql.push("AND saas_invoice_id = ''");
    }
    return await this.customQuery(sql.join(" "), params);
  }

  async bySaasInvoiceId(params) {
    let attrs = params.attrs || "*";
    let sql = [];
    sql.push("SELECT");
    sql.push(attrs);
    sql.push("FROM");
    sql.push(this.model.tableName);
    sql.push("WHERE saas_invoice_id = :saas_invoice_id ");
    return await this.customQuery(sql.join(" "), params);
  }

  async updateTrade(params, t) {
    let sql = "UPDATE st_order_item SET trade_time = :trade_time, trade_status = '00', trade_desc = :trade_desc WHERE id IN (:ids) AND trade_status = '01' ";
    await this.customUpdate(sql, params, t);
  }

  async updateInvoice(params, t) {
    let sql = "UPDATE st_order_item SET saas_invoice_id = :saas_invoice_id WHERE id IN (:ids) ";
    await this.customUpdate(sql, params, t);
  }

  async cancelInvoice(params, t) {
    let sql = "UPDATE st_order_item SET saas_invoice_id = '' WHERE saas_invoice_id = :saas_invoice_id ";
    await this.customUpdate(sql, params, t);
  }

  async countItems(params, t) {
    let sql = "SELECT count(1) AS num FROM st_order_item WHERE order_id = :order_id AND trade_status = :trade_status";
    let list = await this.customQuery(sql, params, t);
    if (!list || list.length == 0) {
      return 0;
    }
    return list[0].num || 0;
  }

  async listByIds(ids, attrs) {
    if (!ids || ids.length == 0) {
      return [];
    }
    let sql = [];
    sql.push("SELECT");
    sql.push(attrs || "*");
    sql.push("FROM");
    sql.push(this.model.tableName);
    sql.push("WHERE id IN (:ids)");
    return await this.customQuery(sql.join(" "), {
      ids: ids
    }) || [];
  }

  async mapByIds(ids, attrs) {
    let result = {};
    if (!ids || ids.length == 0) {
      return result;
    }
    let list = await this.findListByIds(ids, attrs);
    if (!list || list.length == 0) {
      return result;
    }
    for (var item of list) {
      result[item.id] = item;
    }
    return result;
  }

  async genTradeNo(orderId, t) {
    let sql = `UPDATE ${this.model.tableName} SET trade_no = CONCAT('T', id, SUBSTR(order_id, LENGTH(order_id) - 2, 4), SUBSTRING(REPLACE(RAND(),'.',''),3,4)) WHERE order_id = :orderId`;
    return await this.customUpdate(sql, {orderId: orderId}, t);
  }

  async countByCondition(params) {
    var sql = [];
    sql.push("SELECT");
    sql.push("count(1) as num");
    sql.push("FROM");
    sql.push(this.model.tableName);
    sql.push("WHERE deleted_at IS NULL");

    this.setCondition(sql, params);

    let list = await this.customQuery(sql.join(" "), params);
    if (!list || list.length == 0) {
      return 0;
    }
    return list[0].num;
  }

  async listByCondition(params) {
    params.startRow = Number(params.startRow || 0);
    params.pageSize = Number(params.pageSize || 10);
    let sql = [];
    sql.push("SELECT");
    sql.push(params.attrs || "*");
    sql.push("FROM");
    sql.push(this.model.tableName);
    sql.push("WHERE deleted_at IS NULL");

    this.setCondition(sql, params);

    sql.push("ORDER BY created_at DESC");
    sql.push("LIMIT :startRow, :pageSize");
    return await this.customQuery(sql.join(" "), params);
  }

  setCondition(sql, params) {
    if (!params || !sql) {
      return;
    }
    if (params.saas_id) {
      sql.push("AND saas_id = :saas_id");
    }
    if (params.saas_merchant_id) {
      sql.push("AND saas_merchant_id = :saas_merchant_id");
    }
    if (params.order_id) {
      sql.push("AND order_id = :order_id");
    }
    if (params.order_type) {
      sql.push("AND order_type = :order_type");
    }
    if (params.saas_merchant_ids) {
      sql.push("AND saas_merchant_id IN (:saas_merchant_ids)");
    }
    if (params.out_trade_no) {
      sql.push("AND out_trade_no = :out_trade_no");
    }
    if (params.trade_status) {
      sql.push("AND trade_status = :trade_status");
    }
    if (params.acc_name) {
      params.acc_name_like = `%${params.acc_name}%`;
      sql.push("AND acc_name LIKE :acc_name_like");
    }
    if (params.acc_no) {
      sql.push("AND acc_no = :acc_no");
    }
    if (params.credit_code) {
      sql.push("AND credit_code = :credit_code");
    }
    if (params.createBegin) {
      sql.push("AND created_at >= :createBegin");
    }
    if (params.createEnd) {
      sql.push("AND created_at <= :createEnd");
    }
    if (params.unInvoice) {
      sql.push("AND saas_invoice_id = ''");
    }
  }
}

module.exports = StOrderItemDao;
