const system = require("../system");
const redis = require("redis");
const settings = require("../../config/settings");
const bluebird = require("bluebird");
bluebird.promisifyAll(redis);
class RedisClient {
  constructor() {
    this.redisConfig = settings.redis();
    this.client = this.getCreateClientInstance();
    this.subclient = this.client.duplicate();

    this.client.on("error", function (err) {
      //TODO：日志处理
      console.log(err, "..redisClient........error");
    });
    const self = this;
    // 监听回调
    const subExpired = async (err, res) => {
      // 这里需要创建一个新的Redis对象
      // 因为 连接在订阅服务器模式下，只能使用订阅服务器命令
      const sub = self.getCreateClientInstance();
      // 设置事件参数
      const expired_subKey = `__keyevent@${self.redisConfig.db}__:expired`
      sub.subscribe(expired_subKey, function () {
        sub.on('message', async function (channel, msg) {
          try {
            var opResult = msg.split("$$");
            if (!opResult || opResult.length == 0) {
              return;
            }
            if ("SYTXFAIL-SYTXPUBLIC-MSGQ" != opResult[0]) {
              return;
            }
            var lock = await self.enter(opResult[1]);
            if (lock && lock === "1") {
              var opData = await self.zrangeList(opResult[0], opResult[1], opResult[1]);
              if (!opData || opData.length == 0) {
                return;
              }
              var opDataItem = JSON.parse(opData[0]);
              await self.lpushData(opDataItem.queuedName, opDataItem);
              await self.zremRangebyscoreData(opResult[0], opResult[1], opResult[1]);
            }
          } catch (error) {
            //TODO：日志处理
            console.log(error.stack, ".....message......................error...........");
          }

        });
      });
    };
    // 创建监听
    this.client.send_command('config', ['set', 'notify-keyspace-events', 'Ex'], subExpired)

  }
  getDuplicateInstance() {
    return this.subclient;
  }
  getCreateClientInstance() {
    const self = this;
    return redis.createClient({
      host: self.redisConfig.host,
      port: self.redisConfig.port,
      password: self.redisConfig.password,
      db: self.redisConfig.db,
      retry_strategy: function (options) {
        if (options.total_retry_time > 1000 * 60 * 60) {
          return new Error('Retry time exhausted');
        }
        if (options.attempt > 10) {
          return 10000;
        }
        return Math.min(options.attempt * 100, 3000);
      }
    });
  }

  /**
   * 生产供消费者消费的数据
   * @param {*} queuedName 队列名称
   * @param {*} actionBody 业务数据
   */
  lpushData(queuedName, messageBody) {
    messageBody = typeof messageBody === 'object' ? JSON.stringify(messageBody) : messageBody;
    this.client.lpush(queuedName, messageBody, function (err, reply) {
      if (err) {
        return new Error("lpush message error :" + err + "，queuedName：" + queuedName + "，messageBody：" + messageBody);
      } else {
        console.log("lpush message success");
      }
    });
  }
  /**
   * 设置延迟事件redis存储
   * @param {*} key key
   * @param {*} exSeconds 过期时间-单位秒
   * @param {*} value 值
   */
  async setDelayEventData(key, value, exSeconds) {
    value = typeof value === 'object' ? JSON.stringify(value) : value;
    var key = key + "$$" + value;
    this.client.set(key, "delay event", 'EX', exSeconds);
  }

  /**
   * 添加集合数据
   * @param {*} collectionName 集合名称
   * @param {*} score 分数，用于区间查询
   * @param {*} param 参数信息（可以是json）
   */
  async zaddSortedSet(collectionName, score, param) {
    param = typeof param === 'object' ? JSON.stringify(param) : param;
    await this.client.zadd([collectionName + "-SORT", score, param], function (err, data) {
      if (err) {
        return new Error("zaddSortedSet data error :" + err + "，collectionName：" + collectionName + "，param：" + param);
      } else {
        console.log("zaddSortedSet data success");
      }
    });
  }
  /**
   * 获取集合的区间值
   * @param {*} collectionName 集合名称
   * @param {*} minValue 最小值 
   * @param {*} maxValue 最大值
   */
  async zrangeList(collectionName, minValue, maxValue) {
    return await this.client.zrangebyscoreAsync(collectionName + "-SORT", minValue, maxValue);
  }
  /**
   * 删除集合的区间值
   * @param {*} collectionName 集合名称
   * @param {*} minValue 最小值
   * @param {*} maxValue 最大值
   */
  async zremRangebyscoreData(collectionName, minValue, maxValue) {
    return await this.client.zremrangebyscoreAsync(collectionName + "-SORT", minValue, maxValue);
  }
  /**
   * 设置缓存
   * @param {*} key key
   * @param {*} val 值
   * @param {*} t 过期时间，为空则永久存在
   */
  async setWithEx(key, val, t) {
    var p = this.client.setAsync(key, val);
    if (t) {
      this.client.expire(key, t);
    }
    return p;
  }
  /**
 * 获取缓存
 * @param {*} key key
 */
  async getCache(key) {
    return this.client.getAsync(key);
  }
  /**
* 删除缓存
* @param {*} key key
*/
  async delete(key) {
    return this.client.delAsync(key);
  }
  /**
   * 设置业务锁
   * @param {*} lockKey 锁key
   */
  async init(lockKey, lockValue, lockTime) {
    this.client.rpushAsync(lockKey, lockValue || "1");
    this.client.expire(lockKey, lockTime || 6000);
  }
  /**
   * 获取业务锁
   * @param {*} lockKey 锁key
   */
  async enter(lockKey) {
    return this.client.rpopAsync(lockKey);
  }
}
module.exports = RedisClient;
