const system = require('../../../system');
const ServiceBase = require('../../sve.base');
const settings = require('../../../../config/settings');
const moment = require('moment');
const System = require('../../../system');
const pushTx = require('../../../utils/totxClient');
const { encryptStr } = system;

class DeliverService extends ServiceBase {
  constructor() {
    super('delivery', ServiceBase.getDaoName(DeliverService));
    this.cacheinfoDao = system.getObject('db.delivery.cacheinfoDao');
    this.materialDao = system.getObject('db.delivery.materialDao');
    this.statuslogDao = system.getObject('db.bizchance.statuslogDao');
    this.qualificationDao = system.getObject('db.delivery.qualificationDao');
    this.annualreportDao = system.getObject('db.delivery.annualreportDao');
  }
  async temporarySave(pobj) {
    const deliverData = await this.dao.findOne({
      id: pobj.deliver_id,
    });
    if (!deliverData) {
      throw new Error('没有关联的交付单');
    }
    const result = await this.cacheinfoDao.createOrUpdate(pobj);
    return result;
  }

  async findTemporary(pobj) {
    return this.cacheinfoDao.findOne({
      deliver_id: pobj.deliver_id,
    });
  }

  async submitMaterials(pobj) {
    /**
         * 1.此状态下是否可以提交材料
         * 2.推送到腾讯
         * 2.同步暂存数据表
         * 3.存储到材料表
         * 4.更改 交付单流转状态
         */
    const deliverData = await this.dao.findOne({
      id: pobj.deliver_id,
    });
    if (!deliverData || ![
      system.SERVERSESTATUS.COLLECTING,
      system.SERVERSESTATUS.SUBMITING,
      system.SERVERSESTATUS.DISPOSEING,
      system.SERVERSESTATUS.POSTING,
      system.SERVERSESTATUS.SUCCESS,
    ].includes(deliverData.delivery_status)) {
      throw new Error('此交付单不可提交材料');
    }

    // 推送到商城
    await pushTx.submitMaterials(deliverData, JSON.parse(JSON.stringify(pobj.cache_info)));


    return this.db.transaction(async (t) => {
      await this.cacheinfoDao.createOrUpdate(pobj, t);
      await this.materialDao.createOrUpdate(pobj, t);
      if (deliverData.delivery_status === system.SERVERSESTATUS.COLLECTING) {
        await this.dao.updateByWhere({
          delivery_status: system.SERVERSESTATUS.SUBMITING,
        }, {
          id: pobj.deliver_id,
        }, t);
        this.statuslogDao.create({
          flow_type: system.FLOWCODE.DELIVERY,
          flow_id: pobj.deliver_id,
          status_code: system.SERVERSESTATUS.SUBMITING,
          salesman_id: pobj.userid,
          salesman_name: pobj.username,
        });
      }

      return 'SUCCESS';
    });
  }

  async findInfo(pobj) {
    /**
         * 交付单表 关联  材料表(材料基本信息、邮寄信息) 、 资质信息表
         */
    let result = await this.dao.findInfo(pobj);
    if (!result) {
      throw new Error('交付单不可查看');
    }
    result = JSON.parse(JSON.stringify(result));
    if (result.master_source_number) {
      const annualReportData = await this.dao.findOne({
        master_source_number: result.master_source_number,
        id: {
          $ne: result.id,
        },
      });
      if (annualReportData) {
        result.delivery_info.annualReport = true;
      } else {
        result.delivery_info.annualReport = false;
      }
    }
    return result;
  }

  async changeDeliveryStatus(pobj) {
    /**
         * 判断 交付单状态下能否更新
         */
    const deliverData = await this.dao.findOne({
      id: pobj.id,
    });
    if (!deliverData) {
      throw new Error('没有此交付单');
    }
    if (![system.SERVERSESTATUS.SUBMITING, system.SERVERSESTATUS.DISPOSEING, system.SERVERSESTATUS.POSTING].includes(deliverData.delivery_status)) {
      throw new Error('此状态下不可手动更新办理状态');
    }
    let status;
    let txstatus;
    let materialInfo;
    switch (deliverData.delivery_status) {
      case system.SERVERSESTATUS.SUBMITING:
        status = system.SERVERSESTATUS.DISPOSEING;
        txstatus = pushTx.TXSTATUS.DISPOSEING;
        break;
      case system.SERVERSESTATUS.DISPOSEING:
        const material = await this.materialDao.findOne({
          deliver_id: pobj.id,
        });
        if (material) {
          const { proposerInfo: { recipientInfo } } = material;
          materialInfo = {
            recipientInfo: {
              ...recipientInfo,
              phone: encryptStr(recipientInfo.phone),
            }
          }
        }
        status = system.SERVERSESTATUS.POSTING;
        txstatus = pushTx.TXSTATUS.POSTING;
        break;
      case system.SERVERSESTATUS.POSTING:
        status = system.SERVERSESTATUS.SUCCESS;
        txstatus = pushTx.TXSTATUS.SUCCESS;
        break;
    }

    await pushTx.pushChangeOrder(txstatus, deliverData.delivery_code, materialInfo);

    await this.dao.updateByWhere({
      delivery_status: status,
    }, {
      id: pobj.id,
    });
    this.statuslogDao.create({
      flow_type: system.FLOWCODE.DELIVERY,
      flow_id: pobj.id,
      status_code: status,
      salesman_id: pobj.userid,
      salesman_name: pobj.username,
    });
    return 'success';
  }

  // 点击详情 将 订单状态 从 已接单更新为收集材料中
  async toCollecting(pobj) {
    const deliverData = await this.dao.findOne({
      id: pobj.id,
    });
    if (!deliverData) {
      throw new Error('没有此交付单');
    }

    if (system.SERVERSESTATUS.RECEIVED === deliverData.delivery_status) {
      await pushTx.pushChangeOrder(pushTx.TXSTATUS.COLLECTING, deliverData.delivery_code);

      await this.dao.updateByWhere({
        delivery_status: system.SERVERSESTATUS.COLLECTING,
      }, {
        id: pobj.id,
      });
      this.statuslogDao.create({
        flow_type: system.FLOWCODE.DELIVERY,
        flow_id: pobj.id,
        status_code: system.SERVERSESTATUS.COLLECTING,
        salesman_id: pobj.userid,
        salesman_name: pobj.username,
      });
    }
    return 'success';
  }

  async addQualification(pobj) {
    /**
         * 1. 保存资质信息
         * 2. 判断是否有年报服务,有的话改变年报状态
         */
    const deliverData = await this.dao.findOne({
      id: pobj.deliver_id,
    });
    if (!deliverData) {
      throw new Error('查不到交付单');
    }
    if (
      ![system.SERVERSESTATUS.DISPOSEING,
      system.SERVERSESTATUS.SUCCESS,
      system.SERVERSESTATUS.POSTING].includes(deliverData.delivery_status)
    ) {
      throw new Error('该交付单状态下不可提交');
    }
    if ([system.SERVERSESTATUS.SUCCESS, system.SERVERSESTATUS.POSTING, system.SERVERSESTATUS.DISPOSEING].includes(deliverData.delivery_status)) {
      // 给腾讯 发消息
      const STATUS = {
        success: pushTx.TXSTATUS.SUCCESS,
        posting: pushTx.TXSTATUS.POSTING,
        disposeing: pushTx.TXSTATUS.DISPOSEING
      };

      await pushTx.pushChangeOrder(STATUS[deliverData.delivery_status], deliverData.delivery_code, {
        qualification: {
          businessScope: pobj.businessScope,
          businessTypes: pobj.businessTypes,
          certificateNumber: pobj.certificateNumber,
          endAt: pobj.endAt,
          file: pobj.file,
          serviceProject: pobj.serviceProject,
          startAt: pobj.startAt,
          deliver_id: pobj.deliver_id,
        },
      });
      await this.qualificationDao.createOrUpdate(pobj);
      return 'SUCCESS';
    }
    let annualReportData;
    const annualReportArry = [];

    // 年报逻辑
    if (deliverData.master_source_number) {
      annualReportData = await this.dao.findOne({
        master_source_number: deliverData.master_source_number,
        id: {
          $ne: deliverData.id,
        },
      });
      if (annualReportData && annualReportData.delivery_status === System.ANNUALREPORT.RECEIVED) {
        // TODO: let year = annualReportData.delivery_info && annualReportData.delivery_info.numYear || 5
        const year = 5;

        for (let i = new Date().getFullYear(); i < new Date().getFullYear() + year; i++) {
          annualReportArry.push({
            year: i,
            status: System.ANNUALREPORT.WAITDECLARE,
            deliver_id: annualReportData.id,
          });
        }
        // 推送年报信息
        await pushTx.pushChangeOrder(pushTx.TXSTATUS.WAITDECLARE, annualReportData.delivery_code, {
          annualReport: annualReportArry,
        });
      }
    }
    return this.db.transaction(async (t) => {
      await this.qualificationDao.createOrUpdate(pobj, t);

      if (annualReportData && annualReportData.delivery_status === System.ANNUALREPORT.RECEIVED) {
        await this.dao.updateByWhere({
          delivery_status: System.ANNUALREPORT.TAKEEFFECT,
        }, {
          id: annualReportData.id,
        }, t);
        this.statuslogDao.create({
          flow_type: system.FLOWCODE.DELIVERY,
          flow_id: annualReportData.id,
          status_code: System.ANNUALREPORT.TAKEEFFECT,
          salesman_id: pobj.userid,
          salesman_name: pobj.username,
        });
        await this.annualreportDao.bulkCreate(annualReportArry, t);
      }

      return 'success';
    });
  }

  async closeDeliver(pobj) {
    /**
         * 1 关闭资质
         * 2 查询是否有年报信息 关闭
         */
    const deliverData = await this.dao.findOne({
      id: pobj.id,
    });
    if (!deliverData) {
      throw new Error('查不到交付单');
    }
    // if (![system.SERVERSESTATUS.RECEIVED, system.SERVERSESTATUS.COLLECTING].includes(deliverData.delivery_status)) {
    //   throw new Error('该交付单状态下不可提交');
    // }
    // 推送 关闭交付
    await pushTx.pushChangeOrder(pushTx.TXSTATUS.CLOSED, deliverData.delivery_code, {
      close_reason: pobj.close_reason,
    });
    // 推送 退费
    await pushTx.returnPremium(deliverData.delivery_code);
    // 年报逻辑
    let annualReportData;
    if (deliverData.master_source_number) {
      annualReportData = await this.dao.findOne({
        master_source_number: deliverData.master_source_number,
        id: {
          $ne: deliverData.id,
        },
      });
      // 推送 关闭交付 资质
      if (annualReportData) {
        await pushTx.pushChangeOrder(pushTx.TXSTATUS.CLOSED, annualReportData.delivery_code, {
          close_reason: '对应的资质交付关闭了',
        });
        // 推送 年报退费
        await pushTx.returnPremium(annualReportData.delivery_code);
      }
    }
    return this.db.transaction(async (t) => {
      await this.dao.updateByWhere({
        delivery_status: system.SERVERSESTATUS.CLOSED,
        close_reason: pobj.close_reason,
      }, {
        id: pobj.id,
      }, t);
      this.statuslogDao.create({
        flow_type: system.FLOWCODE.DELIVERY,
        flow_id: pobj.id,
        status_code: system.SERVERSESTATUS.CLOSED,
        salesman_id: pobj.userid,
        salesman_name: pobj.username,
      });
      if (annualReportData) {
        await this.dao.updateByWhere({
          delivery_status: system.SERVERSESTATUS.CLOSED,
          close_reason: '对应的资质交付关闭了',
        }, {
          id: annualReportData.id,
        }, t);
        this.statuslogDao.create({
          flow_type: system.FLOWCODE.DELIVERY,
          flow_id: annualReportData.id,
          status_code: system.SERVERSESTATUS.CLOSED,
          salesman_id: pobj.userid,
          salesman_name: pobj.username,
        });
      }
      return 'success';
    });
  }

  async addMail(pobj) {
    const result = await this.dao.findInfo(pobj);
    if (!result) {
      throw new Error('交付单不可查看');
    }
    if (
      ![system.SERVERSESTATUS.DISPOSEING,
      system.SERVERSESTATUS.SUCCESS,
      system.SERVERSESTATUS.POSTING].includes(result.delivery_status)
    ) {
      throw new Error('该状态下不可填写邮寄信息');
    }
    const { material } = result;
    const { proposerInfo } = material;
    proposerInfo.recipientInfo = pobj.recipientInfo;
    const STATUS = {
      disposeing: pushTx.TXSTATUS.DISPOSEING,
      success: pushTx.TXSTATUS.SUCCESS,
      posting: pushTx.TXSTATUS.POSTING,
    };

    await pushTx.pushChangeOrder(STATUS[result.delivery_status], result.delivery_code, {
      recipientInfo: {
        ...pobj.recipientInfo,
        phone: encryptStr(pobj.recipientInfo.phone),
      },
    });

    await this.materialDao.updateByWhere({
      proposerInfo,
    }, {
      id: material.id,
    });
    return "SUCCESS"

    // return this.db.transaction(async (t) => {
    //   await this.materialDao.updateByWhere({
    //     proposerInfo,
    //   }, {
    //     id: material.id,
    //   }, t);

    //   if (system.SERVERSESTATUS.DISPOSEING === result.delivery_status) {
    //     await this.dao.updateByWhere({
    //       delivery_status: system.SERVERSESTATUS.POSTING,
    //     }, {
    //       id: pobj.id,
    //     }, t);

    //     this.statuslogDao.create({
    //       flow_type: system.FLOWCODE.DELIVERY,
    //       flow_id: pobj.id,
    //       status_code: system.SERVERSESTATUS.POSTING,
    //       salesman_id: pobj.userid,
    //       salesman_name: pobj.username,
    //     });
    //   }
    //   return 'success';
    // });
  }

  async findAnnualReportInfo(pobj) {
    let annualReportResult = await this.dao.findAnnualReportInfo(pobj);
    if (!annualReportResult) {
      throw new Error('交付单不可查看');
    }
    if (!annualReportResult.master_source_number) {
      throw new Error('查不到主订单号');
    }
    const data = await this.dao.findOne({
      master_source_number: annualReportResult.master_source_number,
      id: {
        $ne: annualReportResult.id,
      },
    });
    if (!data) {
      throw new Error('查不到对应的交付单');
    }
    const result = await this.dao.findInfo({ ...pobj, id: data.id });
    annualReportResult = JSON.parse(JSON.stringify(annualReportResult));
    const { material: { proposerInfo: { businessLicense } }, qualification } = result;
    annualReportResult.businessLicense = businessLicense;
    annualReportResult.qualification = qualification;
    return annualReportResult;
  }

  async declareReport(pobj) {
    /**
         * 1. 更新 年报信息
         * 2. 判断是否 都已申报完毕
         * 3. 改变 交付单状态
         */
    const annualReportData = await this.annualreportDao.findOne({
      id: pobj.id,
    });
    if (!annualReportData) {
      throw new Error('查不到此年报信息');
    }

    const deliverData = await this.dao.findOne({
      id: annualReportData.deliver_id,
    });
    // 年报逻辑
    let isChange = false;
    if (deliverData && deliverData.delivery_status === system.ANNUALREPORT.TAKEEFFECT) {
      const annualReports = await this.annualreportDao.findAll({
        deliver_id: deliverData.id,
        status: system.ANNUALREPORT.WAITDECLARE,
      });
      const ids = annualReports.map(item => item.id);
      if (ids.length === 1 && ids.includes(annualReportData.id)) {
        isChange = true;
      }
    }

    // 推送年报信息
    await pushTx.pushDeclareReport({ id: pobj.id, file: pobj.file }, deliverData);
    // 修改年报订单状态
    if (isChange) {
      await pushTx.pushChangeOrder(pushTx.TXSTATUS.SUCCESS, deliverData.delivery_code);
    }

    return this.db.transaction(async (t) => {
      await this.annualreportDao.updateByWhere({
        file: pobj.file,
        status: system.ANNUALREPORT.DECLARESUCCESS,
      }, {
        id: annualReportData.id,
      }, t);
      if (isChange) {
        await this.dao.updateByWhere({
          delivery_status: system.ANNUALREPORT.SUCCESS,
        }, {
          id: deliverData.id,
        }, t);
        this.statuslogDao.create({
          flow_type: system.FLOWCODE.DELIVERY,
          flow_id: deliverData.id,
          status_code: System.ANNUALREPORT.SUCCESS,
          salesman_id: pobj.userid,
          salesman_name: pobj.username,
        });
      }
      return 'success';
    });
  }
}
module.exports = DeliverService;
