const system = require("../../../system");
const ServiceBase = require("../../sve.base")

class AccountService extends ServiceBase {
  constructor() {
    super("account", ServiceBase.getDaoName(AccountService));
    this.accounttradeDao = system.getObject("db.account.accounttradeDao");
    this.dictionary = system.getObject("util.dictionary");
  }

  async createAccount(params) {
    let app_id = this.trim(params.app_id);
    let app_data_id = this.trim(params.app_data_id);

    let apps = this.dictionary.getDict("APP", "app_id") || {};
    if (!apps[app_id]) {
      return system.getResult(null, `非法的app_id[${app_id}]`);
    }

    if (!app_data_id) {
      return system.getResult(null, `请输入app_data_id[${app_data_id}]`);
    }

    let account = await this.dao.findOne({app_id: app_id, app_data_id: app_data_id});
    if (account) {
      return system.getResult(null, `app_id[${app_id}]，app_data_id[${app_data_id}]账户已创建，请不要重复创建`);
    }
    account = await this.dao.create({
      autoIncrement: true,
      app_id: app_id,
      app_data_id: app_data_id,
      balance: 0,
    });
    return system.getResultSuccess({account_id: account.id});
  }

  async accountInfo(params) {
    let account = await this.dao.getById(params.account_id);
    return system.getResultSuccess(account);
  }

  async accountRecharge(params) {
    let p = {trade_type: 2};
    p.account_id = params.account_id;
    p.trade_amt = Number(params.trade_amt);
    p.trade_no = this.trim(params.trade_no);
    p.trade_desc = params.trade_desc || "充值";
    try {
      return await this.trade(p);
    } catch (e) {
      console.log(e);
      return system.getResult(null, "交易错误");
    }
  }

  async accountRefund(params) {
    let p = {trade_type: 3};
    p.account_id = params.account_id;
    p.trade_amt = Number(params.trade_amt);
    p.trade_no = this.trim(params.trade_no);
    p.trade_desc = params.trade_desc || "退款";
    try {
      return await this.trade(p);
    } catch (e) {
      console.log(e);
      return system.getResult(null, "交易错误");
    }
  }

  async accountTrade(params) {
    let p = {trade_type: 1};
    p.account_id = params.account_id;
    p.trade_amt = Number(params.trade_amt);
    p.trade_no = this.trim(params.trade_no);
    p.trade_desc = params.trade_desc || "消费";
    try {
      return await this.trade(p);
    } catch (e) {
      console.log(e);
      return system.getResult(null, "交易错误");
    }
  }

  async trade(params) {
    let accountId = params.account_id;
    let trade_no = this.trim(params.trade_no);
    let trade_type = params.trade_type;
    let trade_amt = Number(params.trade_amt || 0);
    if (trade_amt < 0) {
      return system.getResult(null, "金额不能为负数");
    }
    let balance_amt = trade_type == 2 ? trade_amt : -1 * trade_amt;

    let trade_desc = params.trade_desc;
    if(!accountId) {
      return system.getResult(null, "account_id不存在");
    }
    if(!trade_no) {
      return system.getResult(null, "交易流水号不存在");
    }
    if(!trade_desc) {
      return system.getResult(null, "请填写交易描述信息");
    }

    let _trade = await this.accounttradeDao.findOne({
      account_id: accountId,
      trade_type: trade_type,
      trade_no: trade_no,
    });
    if (_trade) {
      return system.getResult(null, `账户[${accountId}]已存在交易流水号[${trade_no}]`);
    }

    let res = await this.db.transaction(async t => {
      //更新oorder订单记录
      let bres = await this.dao.incrementBalance(accountId, balance_amt, t);
      console.log(bres);
      if (bres === 0) {
        return system.getResult(null, "余额不足");
      }

      //创建orderdeliver记录
      await this.accounttradeDao.create({
        account_id: accountId,
        trade_type: trade_type,
        trade_no: trade_no,
        trade_amt: trade_amt,
        trade_desc: trade_desc,
      }, t);
      return system.getResultSuccess();
    });
    return res;
  }

  async accountTradePage(params) {
    if (!params.account_id) {
      return {count: 0, rows: []};
    }
    return await this.tradePageByCondition(params);
  }

  async tradePageByCondition(params) {
    let page = {
      count: 0,
      rows: []
    }
    params.currentPage = Number(params.currentPage || 1);
    params.pageSize = Number(params.pageSize || 10);
    params.startRow = (params.currentPage - 1) * params.pageSize;

    page.count = await this.accounttradeDao.countByCondition(params);
    if (page.count == 0) {
      return system.getResultSuccess(page);
    }
    page.rows = await this.accounttradeDao.listByCondition(params);
    this.dictionary.setRowsName("ACCOUNT_TRADE", page.rows, ["trade_type"]);
    this.handleRowsDate(page.rows, ["created_at", "updated_at"]);
    return system.getResultSuccess(page);
  }
  async test(params) {
    return system.getResultSuccess("test");
  }
}

module.exports = AccountService;
// var task=new UserService();
// task.getUserStatisticGroupByApp().then(function(result){
//   console.log((result));
// }).catch(function(e){
//   console.log(e);
// });