const rabbitmqClient = require("../client/rabbitmq");
const {
  PREFETCH,
  LADDERDELAYOPTIONSL: {
    time,
    lodderExchangeDLX,
    lodderQueueKeyDLX,
    lodderRoutingKeyDLX,
    consumerExchange,
    consumerQueue,
    consumerRoutingKey
  },
  DELAYOPTIONSL: {
    delayExchange,
    delayQueue,
    delayRoutingKey
  }
} = require("../settings")

const { v4: uuidv4 } = require('uuid');

const { request } = require("./axios");

/**
 * 重试机制
 * 初始化 阶梯延迟队列
 * TTL + DLX 
 */
async function initLadderQueue() {
  const channel = await rabbitmqClient.getConfirmChannel();

  // 初始化 阶梯延时队列 
  await channel.assertExchange(lodderExchangeDLX, 'direct', { durable: true });
  for (let val of time) {
    const queue = `${lodderQueueKeyDLX}.${val}`;
    await channel.assertQueue(queue,
      {
        exclusive: false,
        deadLetterExchange: consumerExchange,
        deadLetterRoutingKey: consumerRoutingKey,
        arguments: {
          "x-message-ttl": val * 1000
        }
      });
    await channel.bindQueue(queue, lodderExchangeDLX, `${lodderRoutingKeyDLX}.${val}`);
  }

  // 初始化死信队列
  await channel.assertExchange(consumerExchange, 'direct', { durable: true });
  await channel.assertQueue(consumerQueue, { exclusive: false, });
  await channel.bindQueue(consumerQueue, consumerExchange, consumerRoutingKey);


  console.log("--- 阶梯延迟队列 && 死信队列 初始化成功 ----")
}



/**
 * 重试机制
 * 消费者 监听处理 业务
 */
async function listenCustomerLadderQueue() {
  await initLadderQueue();
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.prefetch(PREFETCH, true);
  await channel.consume(consumerQueue, async (msg) => {
    const content = JSON.parse(msg.content.toString());
    try {
      await request(content);
      channel.ack(msg);
    } catch (err) {
      if (content.retryCount < time.length) {
        await sendRetryMessage({
          ...content,
          retryCount: content.retryCount + 1
        }, `${lodderQueueKeyDLX}.${time[content.retryCount]}`)
        channel.ack(msg);
      } else {
        // 重试次数用完还是失败
        console.log("------ 重试 发送消息 彻底 ERROR ------")
        console.log(msg.content.toString());
        channel.ack(msg);
      }
    }
  },
    { noAck: false }
  );
}


/**
 * 重试机制
 * 发送消息
 * 1. 先入库 消息状态为 未知
 * 2. 发送消息 
 * 3. 将消息状态改为 已发送到mq 
 */
async function sendRetryMessage(message, queue = consumerQueue) {
  if (!message.messageId) {
    message.messageId = uuidv4();
  }
  if (!message.retryCount) {
    message.retryCount = 0;
  }
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.sendToQueue(queue, Buffer.from(JSON.stringify(message)), {
    persistent: true, // 持久化 消息 
    messageId: message.messageId,
    timestamp: new Date().getTime(),
  }, function (err, ok) {
    // console.log(err, ok)
  })
  return {
    messageId: message.messageId
  }
}


/**
 * 延时
 * 初始化 延时队列 
 * 根据 延时队列插件
 */

async function initDelayedQueue() {
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.assertExchange(delayExchange, 'x-delayed-message',
    {
      durable: true,
      arguments: {
        'x-delayed-type': 'direct'
      }
    });
  await channel.assertQueue(delayQueue, { exclusive: false, });
  await channel.bindQueue(delayQueue, delayExchange, delayRoutingKey);
  console.log("--- 延时插件队列 初始化成功 ---")
}

/**
 * 延时
 * 消费者 监听处理 业务
 * 延时插件
 */

async function listenCustomerQueue() {
  await initDelayedQueue();
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.prefetch(PREFETCH, true);

  await channel.consume(delayQueue, async (msg) => {
    let content = JSON.parse(msg.content.toString());
    try {
      await request(content);
      channel.ack(msg);
    } catch (err) {
      console.log("------ 延时 发送消息 彻底 ERROR ------")
      console.log(msg.content.toString());
      channel.ack(msg);
    }
  },
    { noAck: false }
  );
}



/**
 * 延时
 * 发送消息 - 延时功能
 * 1. 先入库 消息状态为 未知
 * 2. 发送消息 
 * 3. 将消息状态改为 已发送到mq 
*/
async function sendDelayMessage(message) {
  if (!message.messageId) {
    message.messageId = uuidv4();
  }
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.publish(delayExchange, delayRoutingKey, Buffer.from(JSON.stringify(message)), {
    persistent: true, // 持久化 消息 
    messageId: message.messageId,
    timestamp: new Date().getTime(),
    headers: {
      'x-delay': message.delayTime * 1000, // 一定要设置，否则无效
    }
  }, function (err, ok) {
  })
  return {
    messageId: message.messageId
  }
}

async function initQueue() {
  await listenCustomerLadderQueue()
  await listenCustomerQueue()
}

module.exports = {
  initQueue,
  sendRetryMessage,
  sendDelayMessage
}
