import rabbitmqClient from "../client/rabbitmq";
import settings from "../settings"
const {
  PREFETCH,
  LADDERDELAYOPTIONSL: {
    time,
    lodderExchangeDLX,
    lodderQueueKeyDLX,
    lodderRoutingKeyDLX,
    consumerExchange,
    consumerQueue,
    consumerRoutingKey
  },
  DELAYOPTIONSL: {
    delayExchange,
    delayQueue,
    delayRoutingKey
  }
} = settings;
import { v4 as uuidv4 } from 'uuid';

import { request } from "./axios";

import logger from "../utils/logger"

import { getNamespace } from 'cls-hooked';

import { Message } from "../model/index"

/**
 * 重试机制
 * 初始化 阶梯延迟队列
 * TTL + DLX 
 */
async function initLadderQueue() {
  const channel = await rabbitmqClient.getConfirmChannel();

  // 初始化 阶梯延时队列 
  await channel.assertExchange(lodderExchangeDLX, 'direct', { durable: true });
  for (let val of time) {
    const queue = `${lodderQueueKeyDLX}.${val}`;
    await channel.assertQueue(queue,
      {
        exclusive: false,
        deadLetterExchange: consumerExchange,
        deadLetterRoutingKey: consumerRoutingKey,
        arguments: {
          "x-message-ttl": val * 1000
        }
      });
    await channel.bindQueue(queue, lodderExchangeDLX, `${lodderRoutingKeyDLX}.${val}`);
  }

  // 初始化死信队列
  await channel.assertExchange(consumerExchange, 'direct', { durable: true });
  await channel.assertQueue(consumerQueue, { exclusive: false, });
  await channel.bindQueue(consumerQueue, consumerExchange, consumerRoutingKey);


  logger.info("--- 阶梯延迟队列 && 死信队列 初始化成功 ----")
}



/**
 * 重试机制
 * 消费者 监听处理 业务
 */
async function listenCustomerLadderQueue() {
  await initLadderQueue();
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.prefetch(PREFETCH, true);
  await channel.consume(consumerQueue, async (msg) => {
    const content = JSON.parse(msg.content.toString());
    try {
      const response = await request(content);
      await Message.updateOne({ messageId: content.messageId }, {
        status: 2,
        $push: {
          resContent: response
        }
      });
      channel.ack(msg);
    } catch (err) {
      await Message.updateOne({ messageId: content.messageId }, {
        status: -2,
        $push: {
          resContent: {
            errMsg: err.message || "未知"
          }
        }
      });
      if (content.retryCount < time.length) {
        try {
          await sendRetryMessage({
            ...content,
            retryCount: content.retryCount + 1
          }, `${lodderQueueKeyDLX}.${time[content.retryCount]}`)
        } catch (err) {
          log.error(err)
        }
        channel.ack(msg);
      } else {
        // 重试次数用完还是失败
        channel.ack(msg);
      }
    }
  },
    { noAck: false }
  );
}


/**
 * 重试机制
 * 发送消息
 * 1. 先入库 消息状态为 未知
 * 2. 发送消息 
 * 3. 将消息状态改为 已发送到mq 
 */
export async function sendRetryMessage(message, queue = consumerQueue) {
  if (!message.headers) {
    message.headers = {}
  }
  if (!message.headers["X-Request-Id"]) {
    const logNameSpace = getNamespace('logger');
    message.headers["X-Request-Id"] = logNameSpace.get("requestId") || uuidv4();
  }

  if (!message.messageId) {
    message.messageId = uuidv4();
    await new Message({
      messageId: message.messageId,
      type: 2,
      reqContent: message,
      status: 0,
    }).save()
  } else {
    await Message.updateOne({ messageId: message.messageId }, {
      messageId: message.messageId,
      type: 2,
      reqContent: message,
      status: 0,
    });
  }

  if (!message.retryCount) {
    message.retryCount = 0;
  }

  const channel = await rabbitmqClient.getConfirmChannel();
  const fun = () => {
    return new Promise((resolve, reject) => {
      channel.sendToQueue(queue, Buffer.from(JSON.stringify(message)), {
        persistent: true, // 持久化 消息 
        messageId: message.messageId,
        timestamp: new Date().getTime(),
      }, async function (err, ok) {
        if (err !== null) {
          await Message.updateOne({ messageId: message.messageId, status: 0 }, {
            status: -1
          });
          logger.error(err);
          reject(new Error(`消息Id [${message.messageId}] 发送mq失败`))
        } else {
          await Message.updateOne({ messageId: message.messageId, status: 0 }, {
            status: 1
          });
          resolve(message.messageId)
        }
      })
    })
  }
  return fun()
}


/**
 * 延时
 * 初始化 延时队列 
 * 根据 延时队列插件
 */

async function initDelayedQueue() {
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.assertExchange(delayExchange, 'x-delayed-message',
    {
      durable: true,
      arguments: {
        'x-delayed-type': 'direct'
      }
    });
  await channel.assertQueue(delayQueue, { exclusive: false, });
  await channel.bindQueue(delayQueue, delayExchange, delayRoutingKey);
  logger.info("--- 延时插件队列 初始化成功 ---")
}

/**
 * 延时
 * 消费者 监听处理 业务
 * 延时插件
 */

async function listenCustomerQueue() {
  await initDelayedQueue();
  const channel = await rabbitmqClient.getConfirmChannel();
  await channel.prefetch(PREFETCH, true);

  await channel.consume(delayQueue, async (msg) => {
    let content = JSON.parse(msg.content.toString());
    try {
      const response = await request(content);
      await Message.updateOne({ messageId: content.messageId }, {
        status: 2,
        $push: {
          resContent: response
        }
      });
      channel.ack(msg);
    } catch (err) {
      await Message.updateOne({ messageId: content.messageId }, {
        status: -2,
        $push: {
          resContent: {
            errMsg: err.message || "未知"
          }
        }
      });
      channel.ack(msg);
    }
  },
    { noAck: false }
  );
}



/**
 * 延时
 * 发送消息 - 延时功能
 * 1. 先入库 消息状态为 未知
 * 2. 发送消息 
 * 3. 将消息状态改为 已发送到mq 
*/
export async function sendDelayMessage(message) {
  if (!message.headers) {
    message.headers = {}
  }
  if (!message.headers["X-Request-Id"]) {
    const logNameSpace = getNamespace('logger');
    message.headers["X-Request-Id"] = logNameSpace.get("requestId") || uuidv4();
  }

  if (!message.messageId) {
    message.messageId = uuidv4();
    await new Message({
      messageId: message.messageId,
      type: 1,
      reqContent: message,
      status: 0,
    }).save();
  } else {
    await Message.updateOne({ messageId: message.messageId }, {
      messageId: message.messageId,
      type: 1,
      reqContent: message,
      status: 0,
    });
  }

  const channel = await rabbitmqClient.getConfirmChannel();
  const fun = () => {
    return new Promise((resolve, reject) => {
      channel.publish(delayExchange, delayRoutingKey, Buffer.from(JSON.stringify(message)), {
        persistent: true, // 持久化 消息 
        messageId: message.messageId,
        timestamp: new Date().getTime(),
        headers: {
          'x-delay': message.delayTime * 1000, // 一定要设置，否则无效
        }
      }, async function (err, ok) {
        if (err !== null) {
          await Message.updateOne({ messageId: message.messageId, status: 0 }, {
            status: -1
          });
          logger.error(err);
          reject(new Error(`消息Id [${message.messageId}] 发送mq失败`))
        } else {
          await Message.updateOne({ messageId: message.messageId, status: 0 }, {
            status: 1
          });
          resolve(message.messageId)
        }
      })
    })
  }
  return fun()
}

export async function initQueue() {
  await listenCustomerLadderQueue()
  await listenCustomerQueue()
}


